<?php
/*--------------------------------------------------------------
   GambioOrdersCount.php 2021-08-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use NumberFormatter;

class GambioOrdersCount extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.orders.count';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Bestellungen',
        self::LANGUAGE_CODE_ENGLISH => 'Orders',
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        Connection $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createText(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdownIncludingToday($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $timespan = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)->value());
        $orders   = $this->connection->createQueryBuilder()
                        ->select('COUNT(date_purchased) AS orders')
                        ->from('orders')
                        ->where('orders_status NOT IN (:excludedOrderIds)')
                        ->andWhere('date_purchased BETWEEN :startDate AND :endDate')
                        ->setParameters([
                                            ':excludedOrderIds' => implode(',', self::EXCLUDED_ORDER_STATUS_IDS),
                                            ':startDate'        => $timespan->startDate()
                                                ->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
                                            ':endDate'          => $timespan->endDate()
                                                ->format(self::QUERY_TIMESPAN_FORMAT_DATE_END),
                                        ])
                        ->execute()
                        ->fetchAll()[0];
        
        return $this->dataFactory->useTextData()->createTextData($this->dataFactory->useTextData()
                                                                     ->createValue($this->numberFormatter->format((int)($orders['orders']
                                                                                                                        ??
                                                                                                                        0))));
    }
}
